# ============================================================
# Home Data for ML Course (Kaggle Learn Users)
# 目的：Public LB で RMSE < 12000 を狙う（小さいほど良い）
#
# このコードの狙い（入門～実戦の“ちょうど良い強さ”）
#  1) 数値 + カテゴリを全部使う（情報を捨てない）
#  2) 欠損はシンプルに補完（数値=中央値 / カテゴリ=最頻値）
#  3) カテゴリは One-Hot に変換（handle_unknownで事故防止）
#  4) LightGBM（GBDT）で非線形・相互作用を拾う
#  5) KFold で複数モデルを学習して test 予測を平均化（安定化）
#  6) よく効く外れ値（超広いのに安い家）を除去
#
# 重要：この入門コンペはスコアが 1万台など“大きい数”で出ます。
#       価格（ドル）空間のRMSEなので正常です。
# ============================================================

import numpy as np
import pandas as pd

from sklearn.model_selection import KFold
from sklearn.compose import ColumnTransformer
from sklearn.pipeline import Pipeline
from sklearn.impute import SimpleImputer
from sklearn.preprocessing import OneHotEncoder
from sklearn.metrics import mean_squared_error

from lightgbm import LGBMRegressor


# ============================================================
# 1) データ読み込み
# ============================================================
train = pd.read_csv("/kaggle/input/home-data-for-ml-course/train.csv")
test  = pd.read_csv("/kaggle/input/home-data-for-ml-course/test.csv")

# 目的変数
y = train["SalePrice"].copy()

# 特徴量
X = train.drop(columns=["SalePrice"]).copy()

# テスト（予測対象）
X_test = test.copy()


# ============================================================
# 2) 外れ値除去（効くことが多い定番）
#    「超広いのに安い」家はモデルを変に引っ張ることがある
# ============================================================
outlier_idx = train[(train["GrLivArea"] > 4000) & (train["SalePrice"] < 300000)].index
X = X.drop(index=outlier_idx)
y = y.drop(index=outlier_idx)


# ============================================================
# 3) 数値列・カテゴリ列を自動判定
# ============================================================
num_cols = X.select_dtypes(include=["int64", "float64"]).columns.tolist()
cat_cols = X.select_dtypes(include=["object"]).columns.tolist()

print("Numeric cols:", len(num_cols), " / Categorical cols:", len(cat_cols))


# ============================================================
# 4) 前処理（ColumnTransformer）
# ------------------------------------------------------------
# 数値：
#   - 欠損を中央値で埋める
# カテゴリ：
#   - 欠損を最頻値で埋める
#   - OneHotEncoder(handle_unknown="ignore")
#     → trainにないカテゴリがtestに出てもエラーにしない
#
# ★堅牢オプション：
#   OneHotEncoder の出力を sparse（疎行列）にする
#   → 特徴量が数百～千超になってもメモリ的に安定しやすい
#   LightGBM は疎行列を扱えるのでOK
# ============================================================
numeric_transformer = Pipeline(steps=[
    ("imputer", SimpleImputer(strategy="median")),
])

categorical_transformer = Pipeline(steps=[
    ("imputer", SimpleImputer(strategy="most_frequent")),
    ("onehot", OneHotEncoder(handle_unknown="ignore", sparse=True)),
])

preprocess = ColumnTransformer(
    transformers=[
        ("num", numeric_transformer, num_cols),
        ("cat", categorical_transformer, cat_cols),
    ],
    remainder="drop",
)


# ============================================================
# 5) LightGBM パラメータ
# ------------------------------------------------------------
# 12000を狙うための「強めだけど過学習しすぎにくい」設定
# - learning_rate 小さめ → 安定
# - num_leaves 中くらい → 表現力確保
# - subsample/colsample → 過学習抑制
# ============================================================
lgb_params = dict(
    n_estimators=4000,
    learning_rate=0.02,
    num_leaves=96,
    subsample=0.8,
    colsample_bytree=0.8,
    random_state=42,
    n_jobs=-1,
)


# ============================================================
# 6) KFold 学習（ここが今回のエラー対策の核）
# ------------------------------------------------------------
# よくあるエラー：
#   "X has 282 features, but LGBMRegressor is expecting 284"
# 原因：
#   foldごとに OneHotEncoder が学習するカテゴリ集合が微妙に違い、
#   生成される特徴量の数が変わることがある。
#
# 対策（堅牢）：
#   ★foldごとに Pipeline を作り直して fit する（再利用しない）
#   → これで「前回のfitの特徴量数」を引きずらず、確実に動く
# ============================================================
kf = KFold(n_splits=5, shuffle=True, random_state=42)

oof = np.zeros(len(X))
test_pred = np.zeros(len(X_test))

for fold, (tr_idx, va_idx) in enumerate(kf.split(X), start=1):
    X_tr, X_va = X.iloc[tr_idx], X.iloc[va_idx]
    y_tr, y_va = y.iloc[tr_idx], y.iloc[va_idx]

    # ★foldごとに新規Pipelineを作る（重要）
    pipe_fold = Pipeline(steps=[
        ("preprocess", preprocess),
        ("model", LGBMRegressor(**lgb_params)),
    ])

    # 学習
    pipe_fold.fit(X_tr, y_tr)

    # 検証予測（OOF）
    pred_va = pipe_fold.predict(X_va)
    oof[va_idx] = pred_va

    # fold RMSE
    rmse_fold = mean_squared_error(y_va, pred_va, squared=False)
    print(f"[Fold {fold}] RMSE: {rmse_fold:.2f}")

    # test予測（fold平均）
    test_pred += pipe_fold.predict(X_test) / kf.n_splits

# 全体CV RMSE（LBと完全一致ではないが、改善の方向性を見るのに有効）
cv_rmse = mean_squared_error(y, oof, squared=False)
print(f"\n[CV] RMSE: {cv_rmse:.2f}")


# ============================================================
# 7) submission.csv を作って保存
# ============================================================
submission = pd.DataFrame({
    "Id": test["Id"],
    "SalePrice": test_pred
})
submission.to_csv("submission.csv", index=False)
print("✅ saved: submission.csv")
